-- create OLTP write workload stored procedures for Microsoft SQL Server

use pcmag
go

-- random inputs: 
--   @randbase: 2..1,000,000
--   @randfourmill: 2..4,000,000
--   @randfivemill: 2..5,000,000

----------
----------

-- procedure oltp_write_01 (randbase)
-- increment randomly chosen updates.p_int by 1 and save old row in update_int_history
-- changes: updates.p_int, update_int_history
-- might use index on: updates.p_key
-- competes with: oltp_read_select_04, oltp_read_select_05, oltp_read_select_06, oltp_write_06
-- checks atomicity, isolation
-- single table 1 row select and update on an int, 1 row insert

create proc oltp_write_01 (@randbase int) as
declare
	@oldint		int
	begin transaction
		select @oldint = updates.p_int
			from updates with (updlock)
			where updates.p_key = @randbase
		update updates set updates.p_int = updates.p_int + 1
			where updates.p_key = @randbase
		if (@@error <> 0) goto on_error
		insert into update_int_history values (@randbase, @oldint, getdate())
		if (@@error <> 0) goto on_error
	commit transaction
	if (@@error <> 0) goto on_error
	return (0)
on_error:
	raiserror ('Error in oltp_write_01', 18, 1)
	rollback transaction
	return (1)
go

--

create proc oltp_write_01_check as
declare
	@checkval1	float,
	@checkval2	float

	select @checkval1 = sum(convert(float,updates.p_int)) from updates
	select @checkval1 = @checkval1 - count(*) from update_int_history 
	select @checkval2 = oltp_write_01_check_value.int_sum from oltp_write_01_check_value

	select "oltp_write_01_check should return two values of 500,000,499,999. The actual values are ", @checkval1, @checkval2
	if (@checkval1 = @checkval2 and @checkval2 = 500000499999) begin
		select "Consistency check passed."
		return 0
	end
	else begin
		raiserror ('**CONSISTENCY CHECK FAILURE** in oltp_write_01.', 18, 1)
		return (1)
	end
go

--

create proc oltp_write_01_undo as
	select "Undoing effects of oltp_write_01"
	update updates 
		set updates.p_int = updates.p_key
		where updates.p_int <> updates.p_key
	truncate table update_int_history
go

----------

-- procedure oltp_write_02 (randbase)
-- increment randomly chosen updates.p_signed by 1 and save old row in update_signed_history
-- changes: updates.p_signed, update_signed_history
-- might use index on: updates.p_key
-- competes with oltp_read_select_04, oltp_read_select_05, oltp_read_select_06, oltp_read_join_03
-- checks atomicity, isolation
-- single table 1 row select and update on an int, 1 row insert

create proc oltp_write_02 (@randbase int) as
declare
	@oldsigned	int
	begin transaction
		select @oldsigned = updates.p_signed 
			from updates with (updlock)
			where updates.p_key = @randbase
		update updates set updates.p_signed = updates.p_signed + 1
			where updates.p_key = @randbase
		if (@@error <> 0) goto on_error
		insert into update_signed_history values (@randbase, @oldsigned, getdate())
		if (@@error <> 0) goto on_error
	commit transaction
	if (@@error <> 0) goto on_error
	return (0)
on_error:
	raiserror ('Error in oltp_write_02', 18, 1)
	rollback transaction
	return (1)
go

--

create proc oltp_write_02_check as
declare
	@checkval1	float,
	@checkval2	float

	select @checkval1 = sum(convert(float,updates.p_signed)) from updates
	select @checkval2 = count(*) from update_signed_history

	select "oltp_write_02_check should return two identical values (the number of times oltp_write_02 has been run). The actual values are ", @checkval1, @checkval2
	if (@checkval1 = @checkval2) begin
		select "Consistency check passed."
		return 0
	end
	else begin
		raiserror ('**CONSISTENCY CHECK FAILURE** in oltp_write_02.', 18, 1)
		return (1)
	end
go

--

create proc oltp_write_02_undo as
	select "Undoing effects of oltp_write_02"
	update updates
		set updates.p_signed = update_signed_history.p_signed
		from update_signed_history, updates
		where update_signed_history.p_key = updates.p_key
		and update_signed_history.p_signed in
			(select min(update_signed_history.p_signed) 
			from update_signed_history, updates
			where update_signed_history.p_key = updates.p_key
			group by update_signed_history.p_key)
	truncate table update_signed_history
go

----------

-- procedure oltp_write_03 (randfivemill)
-- moves a randomly selected row from fivemill to del_history
-- changes: fivemill, del_history
-- might use index on: fivemill.h_key
-- competes with oltp_read_join_03
-- checks atomicity, isolation
-- single row select and insert, single row delete

create proc oltp_write_03 (@randfivemill int) as
	begin transaction
		insert into del_history
			select * from fivemill with (updlock) where fivemill.h_key = @randfivemill
		if (@@error <> 0) goto on_error
		delete from fivemill
			where fivemill.h_key = @randfivemill
		if (@@error <> 0) goto on_error
	commit transaction
	if (@@error <> 0) goto on_error
	return (0)
on_error:
	raiserror ('Error in oltp_write_03', 18, 1)
	rollback transaction
	return (1)
go

--

create proc oltp_write_03_check as
declare
	@checkval1	int,
	@checkval2	int

	select @checkval1 = count(*) from fivemill
	select @checkval2 = count(*) from del_history

	select "oltp_write_03_check should return exactly 5,000,000. The actual value is ", @checkval1 + @checkval2
	if (@checkval1 + @checkval2 = 5000000) begin
		select "Consistency check passed."
		return 0
	end
	else begin
		raiserror ('**CONSISTENCY CHECK FAILURE** in oltp_write_03.', 18, 1)
		return (1)
	end
go

--

create proc oltp_write_03_undo as
	select "Undoing effects of oltp_write_03"
	insert into fivemill select * from del_history
	truncate table del_history
go

----------

-- procedure oltp_write_04 (randfourmill)
-- modifies between 1 and 2 randomly chosen fourmill.t_name and fourmill.t_address values, increments the update counter oltp_write_04_count_updates.num_updates
-- changes: fourmill.t_name, fourmill.t_address, oltp_write_04_count_updates.num_updates
-- might use index on: fourmill.t_key
-- competes with nothing else
-- checks atomicity
-- single table in range select and update on 2 character fields, 1 int field

create proc oltp_write_04 (@randfourmill int) as
declare
	@numupdates	int
	begin transaction
		select @numupdates = count(*) from fourmill with (updlock)
			where (fourmill.t_key between @randfourmill and @randfourmill + 500)
			and (fourmill.t_name <> 'NewName')
		if (@@error <> 0) goto on_error
		update fourmill
			set fourmill.t_name = 'NewName', fourmill.t_address = 'New Address 320B Lakeside Dr., Foster City, CA'
			where (fourmill.t_key between @randfourmill and @randfourmill + 500)
			and (fourmill.t_name <> 'NewName')
		if (@@error <> 0) goto on_error
		update oltp_write_04_count_updates
			set oltp_write_04_count_updates.num_updates = oltp_write_04_count_updates.num_updates + @numupdates
		if (@@error <> 0) goto on_error
	commit transaction
	if (@@error <> 0) goto on_error
	return (0)
on_error:
	raiserror ('Error in oltp_write_04', 18, 1)
	rollback transaction
	return (1)
go

--

create proc oltp_write_04_check as
declare
	@checkval1	int,
	@checkval2	int

	select @checkval1 = count(*) from fourmill
		where fourmill.t_name = 'NewName'
	select @checkval2 = oltp_write_04_count_updates.num_updates from oltp_write_04_count_updates

	select "oltp_write_04_check should return two identical values. The actual values are ", @checkval1, @checkval2
	if (@checkval1 = @checkval2) begin
		select "Consistency check passed."
		return 0
	end
	else begin
		raiserror ('**CONSISTENCY CHECK FAILURE** in oltp_write_04.', 18, 1)
		return (1)
	end
go

--

create proc oltp_write_04_undo as
	select "Undoing effects of oltp_write_04"
-- we won't restore the original data, but we will modify these rows to move them to a different index branch
	update fourmill
		set fourmill.t_name = 'OldName'
		where fourmill.t_name = 'NewName' and fourmill.t_address = 'New Address 320B Lakeside Dr., Foster City, CA'
	update oltp_write_04_count_updates
		set num_updates = 0
go

----------

-- procedure oltp_write_05 (randbase)
-- does an update on 10 contiguous rows of updates.p_decim and then rolls the changes back
-- changes: (nothing)
-- might use index on updates.p_key 
-- competes with oltp_write_06, dss_select_05, dss_select_06, dss_select_07, dss_select_08, dss_select_09
-- checks atomicity, consistency, isolation
-- single table range update and rollback

create proc oltp_write_05 (@randbase int) as
	begin transaction
		update updates with (updlock)
			set updates.p_decim = abs(updates.p_decim) - 2000000001
			where updates.p_key between @randbase and @randbase + 9
		if (@@error <> 0) goto on_error
	rollback transaction
	if (@@error <> 0) goto on_error
	return (0)
on_error:
	raiserror ('Error in oltp_write_05', 18, 1)
	return (1)
go
			
--

create proc oltp_write_05_check as
declare
	@checkval	int

	select @checkval = count(*) from updates
		where updates.p_decim < -1000000000
	select "oltp_write_05_check should return exactly 0. The actual value is ", @checkval

	if (@checkval = 0) begin
		select "Consistency check passed."
		return 0
	end
	else begin
		raiserror ('**CONSISTENCY CHECK FAILURE** in oltp_write_05.', 18, 1)
		return (1)
	end
go

--

create proc oltp_write_05_undo as
	select "Undoing effects of oltp_write_05"
-- procedure is self-undoing
go
